package o1.robots.gui

import o1.robots.tribal.*
import o1.util.*
import o1.gui.Dialog.*
import scala.collection.immutable.SortedMap

////////////////// NOTE TO STUDENTS //////////////////////////
// For the purposes of our course, it’s not necessary
// that you understand or even look at the code in this file.
//////////////////////////////////////////////////////////////

/** The singleton object `TribeLoader` is capable of checking what robot
  * tribes are available in a particular folder, and loading the ones that are.
  *
  * **NOTE TO STUDENTS: In this course, you don’t need to understand how this object
  * works or can be used.** */
object TribeLoader:

  private val StandardTribeNames = Seq("guardian", "tiger", "patroller", "headless", "bunny")
  val Suffix = ".tribe"
  private def tribeName(path: Path) = path.fileName.dropRight(Suffix.length)
  private def isTribeFile(candidate: Path) = candidate.fileName.endsWith(Suffix) && candidate.fileName.length > Suffix.length

  val All = loadTribes
  val BunnyTribe = All.values.flatten.find( _.name == "bunny" )

  type TribeMap = SortedMap[Path, Option[Tribe]]

  private def loadTribes: TribeMap =
    findReadable(tribesDirName).flatMap(loadTribeDir) getOrElse SortedMap()

  private[gui] lazy val tribesDirName = "tribes/"

  private def loadTribeDir(dir: Path): Option[TribeMap] =
    def isStandard(dir: Path) = StandardTribeNames.exists( name => dir.endsWith(name + Suffix) )
    val newestFirstOrder   = Ordering.by( (p: Path) => p.lastModified ).reverse
    val standardTribeOrder = Ordering.by( (p: Path) => StandardTribeNames.indexOf(tribeName(p)) )
    for tribeFiles <- dir.listFiles(isTribeFile) yield
      val (standardTribes, customTribes) = tribeFiles.partition(isStandard)
      val sortedTribes = customTribes.sorted(using newestFirstOrder) ++ standardTribes.sorted(using standardTribeOrder)
      val nameTribePairs = for file <- tribeFiles yield file -> this.readTribe(file)
      SortedMap(nameTribePairs*)(using Ordering.by(sortedTribes.indexOf))

  private def readTribe(path: Path): Option[Tribe] =
    Try(Tribe(tribeName(path), path.toString)) match
      case Failure(roboSpeakProblem: TribeFileException) =>
        display("Problem in the RoboSpeak file " + path.fileName + ":\n\n" + roboSpeakProblem.getMessage, Centered)
        System.err.println(roboSpeakProblem)
        None
      case Failure(otherProblem) =>
        System.err.println(otherProblem)
        throw otherProblem
      case Success(tribe) =>
        Some(tribe)

end TribeLoader

