package o1.robots.tribal

import scala.util.parsing.combinator.JavaTokenParsers

// NOTE TO STUDENTS: You are not required to understand the code in this file.
// It is used by the Tribe class to turn Strings that contain RoboSpeak into
// Line and Instruction objects.

private[tribal] class RoboSpeakGrammar(val program: Tribe) extends JavaTokenParsers:
  import program.Instruction.*

  def parse(input: String): Line =
    this.parseAll(this.line, input) match
      case Success(parsedCode, _) => parsedCode
      case Failure(problem,    _) => throw TribeFileException(this.program, input)
      case Error(problem,      _) => throw TribeFileException(this.program, input)



  def line = labelDef | instruction | directive | empty

  // Instructions:
  def instruction = (
    move        | spin        | uturn       | waitInstr   | switch      |
    talk        | ifempty     | ifnempty    | iffriend    | ifnfriend   |
    ifwall      | ifnwall     | ifrandom    | goto        | callsub     |
    returnInstr | ifeq        | ifneq       | ifgt        | iflt        |
    set         | enemiesnear | friendsnear | foddernear  | fodderleft  |
    shout       | score       | add1        | plus        | minus       |
    enemiesdir  | friendsdir  )                    ^^ ( Line(_, label = None, directive = None) )
  def        move = "move"                         ^^ ( _ => Move() )
  def        spin = "spin"                         ^^ ( _ => Spin() )
  def       uturn = "uturn"                        ^^ ( _ => UTurn() )
  def   waitInstr = "wait"                         ^^ ( _ => Wait() )
  def      switch = "switch"                       ^^ ( _ => Switch() )
  def        talk = "talk"~>param                  ^^ ( line => Talk(line) )
  def     ifempty = "ifempty"~>maybeCall~param     ^^ { case maybeCall~line => IfEmpty  (branchKind(maybeCall), line) }
  def    ifnempty = "ifnempty"~>maybeCall~param    ^^ { case maybeCall~line => IfNEmpty (branchKind(maybeCall), line) }
  def    iffriend = "iffriend"~>maybeCall~param    ^^ { case maybeCall~line => IfFriend (branchKind(maybeCall), line) }
  def   ifnfriend = "ifnfriend"~>maybeCall~param   ^^ { case maybeCall~line => IfNFriend(branchKind(maybeCall), line) }
  def      ifwall = "ifwall"~>maybeCall~param      ^^ { case maybeCall~line => IfWall   (branchKind(maybeCall), line) }
  def     ifnwall = "ifnwall"~>maybeCall~param     ^^ { case maybeCall~line => IfNWall  (branchKind(maybeCall), line) }
  def    ifrandom = "ifrandom"~>maybeCall~param    ^^ { case maybeCall~line => IfRandom (branchKind(maybeCall), line) }
  def        goto = "goto"~>param                  ^^ ( line => Goto(line) )
  def        ifeq = "ifeq"~>param~param~maybeCall~param  ^^ { case n1~n2~maybeCall~line => IfEq (n1, n2, branchKind(maybeCall), line) }
  def       ifneq = "ifneq"~>param~param~maybeCall~param ^^ { case n1~n2~maybeCall~line => IfNEq(n1, n2, branchKind(maybeCall), line) }
  def        ifgt = "ifgt"~>param~param~maybeCall~param  ^^ { case n1~n2~maybeCall~line => IfGT (n1, n2, branchKind(maybeCall), line) }
  def        iflt = "iflt"~>param~param~maybeCall~param  ^^ { case n1~n2~maybeCall~line => IfLT (n1, n2, branchKind(maybeCall), line) }
  def     callsub = "callsub"~>param               ^^ ( line => CallSub(line) )
  def returnInstr = "return"                       ^^ ( _ => Return() )
  def         set = "set"~>slotParam~param         ^^ { case slotName~value => SetSlot(slotFor(slotName), value) }
  def enemiesnear = "enemiesnear"                  ^^ ( _ => EnemiesNear() )
  def friendsnear = "friendsnear"                  ^^ ( _ => FriendsNear() )
  def  foddernear = "foddernear"                   ^^ ( _ => FodderNear() )
  def  fodderleft = "fodderleft"                   ^^ ( _ => FodderLeft() )
  def       score = "score"                        ^^ ( _ => Score() )
  def  enemiesdir = "enemiesdir"~>param            ^^ ( relDir => EnemiesDir(relDir) )
  def  friendsdir = "friendsdir"~>param            ^^ ( relDir => FriendsDir(relDir) )
  def       shout = "shout"~>param                 ^^ ( line => Shout(line) )
  def        add1 = "add1"~>slotParam              ^^ ( slotName => Add1(slotFor(slotName)) )
  def        plus = "plus"~>slotParam~param~param  ^^ { case targetName~value1~value2 => Plus(slotFor(targetName),  value1, value2) }
  def       minus = "minus"~>slotParam~param~param ^^ { case targetName~value1~value2 => Minus(slotFor(targetName), value1, value2) }

  // Parameters of instructions:
  def      param: Parser[String | Int] = intParam | slotParam | labelParam
  def  slotParam = Tribe.Slots.mkString("|").r ^^ ( slotName => slotName )
  def labelParam = raw"\S+".r                  ^^ ( labelName => labelName )
  def   intParam = wholeNumber                 ^^ ( numString => numString.toInt )
  private def  maybeCall = opt("callsub")
  private def branchKind(call: Option[String]): BranchKind =
    if call.isDefined then BranchKind.CallAndReturn else BranchKind.JustGoTo
  private def slotFor(param: String | Int): String = param match
    case name: String => name
    case number: Int  => ""


  // Labels, empty lines, and directives
  def       empty = ""          ^^ ( _ => Line(NoCommand(), None, None) )
  def    labelDef = raw"\S+:".r ^^ ( labelName => Line(LabelDefinition(), Some(labelName.init), None) )
  def   directive = "pacifist"  ^^ ( directiveName => Line(Directive(), None, Some(directiveName)) )

end RoboSpeakGrammar

private[tribal] case class Line(val instruction: Instruction, val label: Option[String], val directive: Option[String])

